<?php
// contact.php — Progressive enhancement + i18n + basic hardening
// SkyNova Cheque System microsite

// ---------- Config ----------
$TO_EMAIL   = 'info@skynova365.com';
$FROM_EMAIL = 'no-reply@skynova365.com';   // يفضّل أن يكون من نفس الدومين
$FROM_NAME  = 'SkyNova Microsite';
$RATE_LIMIT_SECONDS = 30;                  // الحد الأدنى بين رسالتين من نفس الجلسة

// لو أردت SMTP لاحقًا استخدم PHPMailer هنا، مكان واضح للتبديل:
// $USE_SMTP = false; // حالياً false. عندما تفعّله، ضع كود PHPMailer في هذا الموضع.

// ---------- Helpers ----------
function is_ajax(): bool {
  $ajax = trim($_POST['ajax'] ?? '');
  return (
    (isset($_SERVER['HTTP_X_REQUESTED_WITH']) &&
      strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest'
    ) || $ajax === '1'
  );
}

function lang_code(): string {
  $lang = strtolower(trim($_POST['lang'] ?? ''));
  return in_array($lang, ['ar','en'], true) ? $lang : 'ar';
}

function t(string $key, string $lang): string {
  $m = [
    'ar' => [
      'ok'        => 'تم استلام رسالتك، سنعاود الاتصال بك قريبًا.',
      'bot'       => 'تم الاشتباه في سبام.',
      'required'  => 'يرجى تعبئة الحقول المطلوبة.',
      'email'     => 'البريد الإلكتروني غير صالح.',
      'ratelimit' => 'تم إرسال رسالة حديثًا. يرجى المحاولة بعد قليل.',
      'server'    => 'تعذر إرسال البريد من الخادم. يفضل تفعيل SMTP.',
      'exception' => 'حدث خطأ بالخادم: ',
    ],
    'en' => [
      'ok'        => 'Thanks! Your message has been received.',
      'bot'       => 'Spam suspected.',
      'required'  => 'Please fill in the required fields.',
      'email'     => 'Invalid email address.',
      'ratelimit' => 'You recently sent a message. Please try again shortly.',
      'server'    => 'Server could not send email. SMTP is recommended.',
      'exception' => 'Server error: ',
    ]
  ];
  return $m[$lang][$key] ?? $key;
}

function respond_json($ok, $msg=null, $http=200) {
  header('Content-Type: application/json; charset=utf-8');
  http_response_code($http);
  echo json_encode(['ok'=>$ok, 'msg'=>$msg], JSON_UNESCAPED_UNICODE);
  exit;
}

function redirect_back($ok, $lang, $why='') {
  // اختَر الصفحة حسب اللغة
  $target = ($lang === 'en') ? 'index-en.html' : 'index-ar.html';
  $sep = (strpos($target, '?') === false) ? '?' : '&';
  // نحتفظ بسبب داخلي للتشخيص (اختياري للعرض لاحقاً)
  header('Location: ' . $target . $sep . 'sent=' . ($ok ? '1' : '0') . ($why ? '&why='.rawurlencode($why) : ''));
  exit;
}

function safe_line(string $s, int $max = 200): string {
  // إزالة أسطر جديدة لمنع حقن الترويسات + تقصير
  $s = str_replace(["\r","\n"], [' ',' '], trim($s));
  return mb_substr($s, 0, $max, 'UTF-8');
}

function valid_email(string $email): bool {
  // منع حقن الترويسات
  if (preg_match('/[\r\n]/', $email)) return false;
  return (bool) filter_var($email, FILTER_VALIDATE_EMAIL);
}

// ---------- Rate limit ----------
session_start();
if (!isset($_SESSION['last_send'])) $_SESSION['last_send'] = 0;

// ---------- Main ----------
try {
  $lang    = lang_code();
  $name    = safe_line($_POST['name']    ?? '', 120);
  $company = safe_line($_POST['company'] ?? '', 120);
  $email   = safe_line($_POST['email']   ?? '', 160);
  $phone   = safe_line($_POST['phone']   ?? '', 80);
  $message = trim($_POST['message'] ?? '');
  $website = trim($_POST['website'] ?? ''); // honeypot

  $isAjax  = is_ajax();

  // Honeypot
  if ($website !== '') {
    $msg = t('bot', $lang);
    if ($isAjax) respond_json(false, $msg, 400);
    redirect_back(false, $lang, 'bot');
  }

  // Required
  if ($name === '' || $email === '' || $message === '') {
    $msg = t('required', $lang);
    if ($isAjax) respond_json(false, $msg, 400);
    redirect_back(false, $lang, 'required');
  }

  // Email
  if (!valid_email($email)) {
    $msg = t('email', $lang);
    if ($isAjax) respond_json(false, $msg, 400);
    redirect_back(false, $lang, 'email');
  }

  // Rate limit
  if (time() - (int)$_SESSION['last_send'] < $RATE_LIMIT_SECONDS) {
    $msg = t('ratelimit', $lang);
    if ($isAjax) respond_json(false, $msg, 429);
    redirect_back(false, $lang, 'ratelimit');
  }

  // Build email
  $subject = 'New enquiry — SkyNova Cheque System';
  $body  = "New enquiry from SkyNova Cheque System microsite:\r\n\r\n";
  $body .= "Name:    {$name}\r\n";
  $body .= "Company: {$company}\r\n";
  $body .= "Email:   {$email}\r\n";
  $body .= "Phone:   {$phone}\r\n";
  $body .= "Lang:    {$lang}\r\n";
  $body .= "IP:      " . ($_SERVER['REMOTE_ADDR'] ?? 'unknown') . "\r\n";
  $body .= "UA:      " . ($_SERVER['HTTP_USER_AGENT'] ?? 'unknown') . "\r\n\r\n";
  $body .= "Message:\r\n{$message}\r\n";

  $headers  = "MIME-Version: 1.0\r\n";
  $headers .= "Content-type: text/plain; charset=utf-8\r\n";
  $headers .= "From: {$FROM_NAME} <{$FROM_EMAIL}>\r\n";
  $headers .= "Reply-To: {$email}\r\n";

  // --- Send (mail()) ---
  // ملاحظة: كثير من الاستضافات تحد/تعطل mail(). لو فشل، فعّل SMTP عبر PHPMailer.
  $sent = @mail($TO_EMAIL, $subject, $body, $headers);

  if ($sent) {
    $_SESSION['last_send'] = time();
    $msg = t('ok', $lang);
    if ($isAjax) respond_json(true, $msg, 200);
    redirect_back(true, $lang);
  } else {
    $msg = t('server', $lang);
    if ($isAjax) respond_json(false, $msg, 500);
    redirect_back(false, $lang, 'server-mail');
  }

} catch (Throwable $e) {
  $lang = lang_code(); // محاولة تحديد اللغة حتى في حال الاستثناء
  $msg = t('exception', $lang) . $e->getMessage();
  if (is_ajax()) respond_json(false, $msg, 500);
  redirect_back(false, $lang, 'exception');
}
